<?php

namespace App\Http\Controllers\Siswa;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\PeminjamanAlat;
use App\Models\PeminjamanDetail;
use App\Models\Barang;

class SiswaPeminjamanController extends Controller
{
    public function index()
    {
        // ✅ TAMBAHKAN: Session check
        if (!session('siswa_id')) {
            return redirect()->route('login')->with('error', 'Silakan login terlebih dahulu.');
        }
        
        $siswa_id = session('siswa_id');
        
        // ✅ PERBAIKI: Hanya ambil data siswa yang login
        $peminjaman = PeminjamanAlat::where('peminjam_id', $siswa_id)
            ->where('peminjam_role', 'siswa')
            ->orderBy('created_at', 'desc')
            ->get();
            
        return view('siswa.peminjaman.index', compact('peminjaman'));
    }

    public function create()
    {
        // ✅ TAMBAHKAN: Session check
        if (!session('siswa_id')) {
            return redirect()->route('login');
        }
        
        $barang = Barang::where('stok', '>', 0)->get(); // ✅ Hanya barang tersedia
        return view('siswa.peminjaman.create', compact('barang'));
    }

    public function store(Request $request)
    {
        // ✅ TAMBAHKAN: Session check
        if (!session('siswa_id')) {
            return redirect()->route('login');
        }
        
        // ✅ VALIDASI sederhana
        if (empty($request->barang_id)) {
            return back()->with('error', 'Pilih minimal satu barang.');
        }
        
        // ✅ Buat peminjaman
        $peminjaman = PeminjamanAlat::create([
            'kode_peminjaman' => 'PINJ-' . date('Ymd') . '-' . rand(1000, 9999),
            'peminjam_id' => session('siswa_id'),
            'peminjam_role' => 'siswa',
            'tanggal_pinjam' => $request->tanggal_pinjam,
            'tanggal_kembali' => $request->tanggal_kembali,
            'alasan' => $request->alasan,
            'status' => 'pending',
            'total_barang' => count($request->barang_id),
        ]);

        // ✅ Simpan detail peminjaman
        foreach ($request->barang_id as $index => $barangId) {
            PeminjamanDetail::create([
                'peminjaman_id' => $peminjaman->id,
                'id_barang' => $barangId,
                'jumlah' => $request->jumlah[$index] ?? 1,
                'kondisi_pinjam' => 'Baik',
                'keterangan' => $request->keterangan[$index] ?? '',
            ]);
            
            // ✅ Kurangi stok barang (optional)
            $barang = Barang::find($barangId);
            if ($barang && $barang->stok > 0) {
                $barang->decrement('stok', $request->jumlah[$index] ?? 1);
            }
        }

        return redirect()->route('siswa.peminjaman.index')
            ->with('success', 'Peminjaman berhasil diajukan.');
    }

    public function detail($id)
    {
        // ✅ TAMBAHKAN: Session check
        if (!session('siswa_id')) {
            return redirect()->route('login');
        }
        
        $siswa_id = session('siswa_id');
        
        // ✅ PERBAIKI: Hanya boleh melihat peminjaman sendiri
        $peminjaman = PeminjamanAlat::with('detail.barang')
            ->where('id', $id)
            ->where('peminjam_id', $siswa_id)
            ->where('peminjam_role', 'siswa')
            ->firstOrFail();
            
        return view('siswa.peminjaman.detail', compact('peminjaman'));
    }
    
    // ✅ TAMBAHKAN: Method untuk cancel peminjaman
    public function cancel($id)
    {
        if (!session('siswa_id')) {
            return redirect()->route('login');
        }
        
        $siswa_id = session('siswa_id');
        
        $peminjaman = PeminjamanAlat::where('id', $id)
            ->where('peminjam_id', $siswa_id)
            ->where('peminjam_role', 'siswa')
            ->where('status', 'pending')
            ->first();
            
        if ($peminjaman) {
            // Kembalikan stok barang
            foreach ($peminjaman->detail as $detail) {
                $barang = Barang::find($detail->id_barang);
                if ($barang) {
                    $barang->increment('stok', $detail->jumlah);
                }
            }
            
            $peminjaman->update(['status' => 'dibatalkan']);
            return back()->with('success', 'Peminjaman berhasil dibatalkan.');
        }
        
        return back()->with('error', 'Peminjaman tidak ditemukan atau tidak bisa dibatalkan.');
    }
}