<?php

namespace App\Http\Controllers\Siswa;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\PeminjamanAlat;
use App\Models\PeminjamanDetail;
use App\Models\Barang;
use Illuminate\Support\Facades\DB;

class SiswaPeminjamanController extends Controller
{
    public function index()
    {
        if (!session('siswa_id')) {
            return redirect()->route('login');
        }

        $peminjaman = PeminjamanAlat::where('peminjam_id', session('siswa_id'))
            ->where('peminjam_role', 'siswa')
            ->orderBy('created_at', 'desc')
            ->get();

        return view('siswa.peminjaman.index', compact('peminjaman'));
    }

    public function create()
    {
        if (!session('siswa_id')) {
            return redirect()->route('login');
        }

        // HANYA FILTER TAMPILAN
        $barang = Barang::where('stok', '>', 0)->get();

        return view('siswa.peminjaman.create', compact('barang'));
    }

    public function store(Request $request)
    {
        if (!session('siswa_id')) {
            return redirect()->route('login');
        }

        if (empty($request->barang_id)) {
            return back()->with('error', 'Pilih minimal satu barang');
        }

        DB::beginTransaction();

        try {
            // VALIDASI STOK (TANPA MENGURANGI)
            foreach ($request->barang_id as $i => $barangId) {
                $barang = Barang::findOrFail($barangId);
                $jumlah = $request->jumlah[$i] ?? 1;

                if ($barang->stok < $jumlah) {
                    DB::rollBack();
                    return back()->with(
                        'error',
                        'Stok barang "' . $barang->nama . '" tidak mencukupi'
                    );
                }
            }

            $peminjaman = PeminjamanAlat::create([
                'kode_peminjaman' => 'PINJ-' . date('Ymd') . '-' . rand(1000, 9999),
                'peminjam_id' => session('siswa_id'),
                'peminjam_role' => 'siswa',
                'tanggal_pinjam' => $request->tanggal_pinjam,
                'tanggal_kembali' => $request->tanggal_kembali,
                'alasan' => $request->alasan,
                'status' => 'pending',
                'total_barang' => count($request->barang_id),
            ]);

            foreach ($request->barang_id as $i => $barangId) {
                PeminjamanDetail::create([
                    'peminjaman_id' => $peminjaman->id,
                    'id_barang' => $barangId,
                    'jumlah' => $request->jumlah[$i] ?? 1,
                    'kondisi_pinjam' => 'Baik',
                    'keterangan' => $request->keterangan[$i] ?? null,
                ]);
            }

            DB::commit();

            return redirect()
                ->route('siswa.peminjaman.index')
                ->with('success', 'Peminjaman berhasil diajukan dan menunggu persetujuan guru');
        } catch (\Throwable $e) {
            DB::rollBack();
            return back()->with('error', 'Terjadi kesalahan sistem');
        }
    }

    public function detail($id)
    {
        if (!session('siswa_id')) {
            return redirect()->route('login');
        }

        $peminjaman = PeminjamanAlat::with('detail.barang')
            ->where('id', $id)
            ->where('peminjam_id', session('siswa_id'))
            ->where('peminjam_role', 'siswa')
            ->firstOrFail();

        return view('siswa.peminjaman.detail', compact('peminjaman'));
    }

    public function cancel($id)
    {
        if (!session('siswa_id')) {
            return redirect()->route('login');
        }

        $peminjaman = PeminjamanAlat::where('id', $id)
            ->where('peminjam_id', session('siswa_id'))
            ->where('peminjam_role', 'siswa')
            ->where('status', 'pending')
            ->first();

        if (!$peminjaman) {
            return back()->with('error', 'Peminjaman tidak dapat dibatalkan');
        }

        // ❌ TIDAK ADA LOGIKA STOK DI SINI
        $peminjaman->update([
            'status' => 'dibatalkan'
        ]);

        return back()->with('success', 'Peminjaman berhasil dibatalkan');
    }

    public function return($id)
    {
        if (!session('siswa_id')) {
            return redirect()->route('login');
        }

        DB::beginTransaction();

        try {
            $peminjaman = PeminjamanAlat::with('detail.barang')
                ->where('id', $id)
                ->where('peminjam_id', session('siswa_id'))
                ->where('peminjam_role', 'siswa')
                ->where('status', 'dipinjam')
                ->first();

            if (!$peminjaman) {
                return back()->with('error', 'Peminjaman tidak dapat dikembalikan');
            }

            // 🔁 Kembalikan stok
            foreach ($peminjaman->detail as $detail) {
                $detail->barang->increment('stok', $detail->jumlah);

                // opsional: simpan kondisi kembali
                $detail->update([
                    'kondisi_kembali' => 'Baik'
                ]);
            }

            // 🔄 Update status
            $peminjaman->update([
                'status' => 'dikembalikan'
            ]);

            DB::commit();

            return redirect()
                ->route('siswa.peminjaman.index')
                ->with('success', 'Barang berhasil dikembalikan');
        } catch (\Throwable $e) {
            DB::rollBack();
            return back()->with('error', 'Terjadi kesalahan saat pengembalian');
        }
    }

    public function ambil($id)
    {
        if (!session('siswa_id')) {
            return redirect()->route('login');
        }

        $peminjaman = PeminjamanAlat::where('id', $id)
            ->where('peminjam_id', session('siswa_id'))
            ->where('peminjam_role', 'siswa')
            ->where('status', 'disetujui')
            ->firstOrFail();

        // ❗ TIDAK ADA LOGIKA STOK DI SINI
        $peminjaman->update([
            'status' => 'dipinjam'
        ]);

        return back()->with('success', 'Barang berhasil diambil');
    }



}
